/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file TIME/TIME.c                                                          */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-07                                                           */
/*                                                                            */
/* @brief This file contains functions to compute date and time.              */
/******************************************************************************/

/* *************************** STANDARD INCLUDES **************************** */

#include <math.h>

/* **************************** CUSTOM INCLUDES ***************************** */

#include "TIME.h"

/* ******************************** DEFINES ********************************* */

#define C_YEAR_MOD_4   (  4)
#define C_YEAR_MOD_100 (100)
#define C_YEAR_MOD_400 (400)
#define C_LEAPTIME_HOR ( 23)
#define C_LEAPTIME_MIN ( 59)
#define C_LEAPTIME_SEC ( 60)

/* *************************** TYPES DECLARATION **************************** */

typedef enum
{
    E_LEAPSEC_1972_06_30 = 0,
    E_LEAPSEC_1972_12_31,
    E_LEAPSEC_1973_12_31,
    E_LEAPSEC_1974_12_31,
    E_LEAPSEC_1975_12_31,
    E_LEAPSEC_1976_12_31,
    E_LEAPSEC_1977_12_31,
    E_LEAPSEC_1978_12_31,
    E_LEAPSEC_1979_12_31,
    E_LEAPSEC_1981_06_30,
    E_LEAPSEC_1982_06_30,
    E_LEAPSEC_1983_06_30,
    E_LEAPSEC_1985_06_30,
    E_LEAPSEC_1987_12_31,
    E_LEAPSEC_1989_12_31,
    E_LEAPSEC_1990_12_31,
    E_LEAPSEC_1992_06_30,
    E_LEAPSEC_1993_06_30,
    E_LEAPSEC_1994_06_30,
    E_LEAPSEC_1995_12_31,
    E_LEAPSEC_1997_06_30,
    E_LEAPSEC_1998_12_31,
    E_LEAPSEC_2005_12_31,
    E_LEAPSEC_2008_12_31,
    E_LEAPSEC_2012_06_30,
    E_LEAPSEC_2015_06_30,
    E_LEAPSEC_2016_12_31,
    E_NB_LEAPSECONDS
} tTIME_LeapSeconds;

/* ******************************* CONSTANTS ******************************** */

const char C_TIME_CALENDAR_LONGNAME[E_NB_CALENDAR][C_TIME_CALENDAR_NAME_MAXLENGTH] = {"<UNDEFINED>", "JULIAN", "GREGORIAN", "UTC"};
const char C_TIME_CALENDAR_SHRTNAME[E_NB_CALENDAR][C_TIME_CALENDAR_NAME_MAXLENGTH] = {"UNK"        , "JUL"   , "GRE"      , "UTC"};

static const long C_NUMBER_OF_DAYS_PER_MONTH_NOTBISSEXTILE[E_NB_MONTHS] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
static const long C_NUMBER_OF_DAYS_PER_MONTH_BISSEXTILE   [E_NB_MONTHS] = {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

static const tTIME_Timestamp C_LEAPSECONDS[E_NB_LEAPSECONDS] =
{
    78796800  , /* E_LEAPSEC_1972_06_30 */
    94694400  , /* E_LEAPSEC_1972_12_31 */
    126230400 , /* E_LEAPSEC_1973_12_31 */
    157766400 , /* E_LEAPSEC_1974_12_31 */
    189302400 , /* E_LEAPSEC_1975_12_31 */
    220924800 , /* E_LEAPSEC_1976_12_31 */
    252460800 , /* E_LEAPSEC_1977_12_31 */
    283996800 , /* E_LEAPSEC_1978_12_31 */
    315532800 , /* E_LEAPSEC_1979_12_31 */
    362793600 , /* E_LEAPSEC_1981_06_30 */
    394329600 , /* E_LEAPSEC_1982_06_30 */
    425865600 , /* E_LEAPSEC_1983_06_30 */
    489024000 , /* E_LEAPSEC_1985_06_30 */
    567993600 , /* E_LEAPSEC_1987_12_31 */
    631152000 , /* E_LEAPSEC_1989_12_31 */
    662688000 , /* E_LEAPSEC_1990_12_31 */
    709948800 , /* E_LEAPSEC_1992_06_30 */
    741484800 , /* E_LEAPSEC_1993_06_30 */
    773020800 , /* E_LEAPSEC_1994_06_30 */
    820454400 , /* E_LEAPSEC_1995_12_31 */
    867715200 , /* E_LEAPSEC_1997_06_30 */
    883612800 , /* E_LEAPSEC_1998_12_31 */
    1136073600, /* E_LEAPSEC_2005_12_31 */
    1230768000, /* E_LEAPSEC_2008_12_31 */
    1341100800, /* E_LEAPSEC_2012_06_30 */
    1435708800, /* E_LEAPSEC_2015_06_30 */
    1483228800  /* E_LEAPSEC_2016_12_31 */
};

/* **************************** GLOBAL VARIABLES **************************** */

/* **************************** STATIC VARIABLES **************************** */

/* ********************** STATIC FUNCTIONS DECLARATION ********************** */

/* ************************* FUNCTIONS DECLARATION ************************** */

/* ********************** STATIC FUNCTIONS DEFINITION *********************** */

/******************************************************************************/
/* @function __TIME_IsBissetile                                               */
/*                                                                            */
/* @brief Verifies if a year is bissextile according to the calendar used.    */
/* @param [in] calendar Calendar used                                         */
/* @param [in] year Year to test for bissextility                             */
/* @retval 'true' if the year is bissextile ; 'false' otherwise               */
/* @req SYS_REQ-0503-001 : Calcul de l’horodatage > Bissextile condition      */
/******************************************************************************/
static bool __TIME_IsBissetile(const tTIME_Calendar calendar, unsigned int year)
{
    switch(calendar)
    {
        case E_CALENDAR_JULIAN:
            return ((year % C_YEAR_MOD_4) == 0) ? (true) : (false);

        case E_CALENDAR_GREGORIAN:
        case E_CALENDAR_UTC:
            if((year % C_YEAR_MOD_400) == 0)
            {
                return true;
            }
            else if (((year % C_YEAR_MOD_4) == 0) && (year % C_YEAR_MOD_100 != 0))
            {
                return true;
            }
            return false;

        default:
            return false;
    }
}

/******************************************************************************/
/* @function __TIME_ComputeDateBasedOnEpoch                                   */
/*                                                                            */
/* @brief Computes the current date and time based on the current timestamp   */
/*        and the calendar reference date.                                    */
/* @param [in] referenceDate Reference date in specified calendar             */
/* @param [in] timestamp Current timestamp                                    */
/* @retval Current date and time in specified calendar                        */
/* @req SYS_REQ-0503-001 : Calcul de l’horodatage > Generic computation       */
/******************************************************************************/
static tTIME_CalendarDate __TIME_ComputeDateBasedOnEpoch
(
    const tTIME_CalendarDate * const referenceDate,
    const tTIME_Timestamp timestamp
)
{
    tTIME_CalendarDate currentDate;
    long numberOfDaysElapsedSinceEpoch;
    long timeInDay;
    long ongoingYear;
    long nbDaysRemaining;
    long nbDaysInOngoingYear;
    const long * nbDaysPerMonthInOngoingYear;
    long timeRemainingInDay;
    long firstYear = true;
    long daysOffset;
    bool end = false;
    tTIME_Months month;

    currentDate.calendar = referenceDate->calendar;

    /* ====================================================================== */
    /* If the calendar is invalid, returns an invalid datetime.               */
    /* ====================================================================== */
    if((currentDate.calendar <= E_CALENDAR_UNKNOWN) || (currentDate.calendar >= E_NB_CALENDAR))
    {
        currentDate.validity = false;
        currentDate.year     = C_UNDEFINED_DATE_YEAR;
        currentDate.month    = C_UNDEFINED_DATE_MONTH;
        currentDate.day      = C_UNDEFINED_DATE_DAY;
        currentDate.hour     = C_UNDEFINED_DATE_HOUR;
        currentDate.minute   = C_UNDEFINED_DATE_MINUTE;
        currentDate.second   = C_UNDEFINED_DATE_SECOND;
        return currentDate;
    }

    /* ====================================================================== */
    /* If the calendar is valid, computes the current datetime.               */
    /* ====================================================================== */

    /* The number of days elapsed since Epoch is the number of seconds        */
    /* elapsed since epoch (timestamp) divided by the number of seconds in a  */
    /* day.                                                                   */
    numberOfDaysElapsedSinceEpoch = (long)floorl((double)timestamp / C_NB_SEC_IN_DAY);

    /* The time in day (number of seconds in day) is the timestamp (number of */
    /* seconds since Epoch) minus the cumulated number of seconds of the      */
    /* previous complete days elapsed.                                        */
    timeInDay = (long)(timestamp - (numberOfDaysElapsedSinceEpoch * C_NB_SEC_IN_DAY));

    ongoingYear = referenceDate->year;
    nbDaysRemaining = numberOfDaysElapsedSinceEpoch;

    /* ---------------------------------------------------------------------- */
    /* Date computation                                                       */
    /* ---------------------------------------------------------------------- */
    do
    {
        if(__TIME_IsBissetile(currentDate.calendar, ongoingYear))
        {
            nbDaysInOngoingYear = C_NB_DAYS_BISSEXTILE;
            nbDaysPerMonthInOngoingYear = C_NUMBER_OF_DAYS_PER_MONTH_BISSEXTILE;
        }
        else
        {
            nbDaysInOngoingYear = C_NB_DAYS_NOTBISSEXTILE;
            nbDaysPerMonthInOngoingYear = C_NUMBER_OF_DAYS_PER_MONTH_NOTBISSEXTILE;
        }

        /* As the first year of search is the reference year, it is possible  */
        /* that the reference date is not the 1st of January. In that case,   */
        /* the current date is the number of seconds elapsed since this       */
        /* reference date, with an offset on first day of year. The function  */
        /* then computes the number of days elapsed since the beginning of    */
        /* the reference year (1st of January) and the reference day. This    */
        /* number of days is an offset applied on the current date search for */
        /* the first year.                                                    */
        daysOffset = 0;
        if(firstYear)
        {
            for(month = 0 ; month < referenceDate->month ; ++month)
            {
                daysOffset += nbDaysPerMonthInOngoingYear[month];
            }
            daysOffset += referenceDate->day;
            firstYear = false;
        }

        nbDaysRemaining += daysOffset;

        /* If the number of days remaining before the current date is less    */
        /* than the number of days in ongoing year, the current date is in    */
        /* the ongoing year.                                                  */
        if(nbDaysRemaining < nbDaysInOngoingYear)
        {
            currentDate.year = ongoingYear;
            for(month = 0 ; month < E_NB_MONTHS; ++month)
            {
                if(nbDaysRemaining < nbDaysPerMonthInOngoingYear[month])
                {
                    currentDate.month = month;
                    currentDate.day = nbDaysRemaining;
                    end = true;
                    break;
                }
                else
                {
                    nbDaysRemaining -= nbDaysPerMonthInOngoingYear[month];
                }
            }
        }
        /* Otherwise, if the number of days remaining before the current date */
        /* is greater than the number of days in the ongoing year, the        */
        /* current date is not in the ongoing year and the search continues   */
        /* on next years.                                                     */
        else
        {
            nbDaysRemaining -= nbDaysInOngoingYear;
            ++ongoingYear;
        }
    }
    while(!end);

    /* ---------------------------------------------------------------------- */
    /* Time computation                                                       */
    /* ---------------------------------------------------------------------- */
    timeRemainingInDay = timeInDay;

    /* The current hour is the number of seconds elapsed in the day divided   */
    /* by the number of seconds in an hour.                                   */
    currentDate.hour = (long)floorl((double)timeRemainingInDay / (double)C_NB_SEC_IN_HOUR);
    timeRemainingInDay -= currentDate.hour * C_NB_SEC_IN_HOUR;

    /* The current minute is the number of seconds elapsed in the current     */
    /* hour divided by the number of seconds in a minute.                     */
    currentDate.minute = (long)floorl((double)timeRemainingInDay / (double)C_NB_SEC_IN_MIN);
    timeRemainingInDay -= currentDate.minute * C_NB_SEC_IN_MIN;

    /* The current second is the number of seconds elapsed in the current     */
    /* minute.                                                                */
    currentDate.second = timeRemainingInDay;

    currentDate.validity = true;
    return currentDate;
}

/* ************************** FUNCTIONS DEFINITION ************************** */

/******************************************************************************/
/* @function TIME_ComputeDate                                                 */
/*                                                                            */
/* @brief Computes the current date and time.                                 */
/* @param [in] calendar Calendar used by the system                           */
/* @param [in] timestamp Current timestamp                                    */
/* @retval Current date and time in specified calendar                        */
/* @req SYS_REQ-0503-001 : Calcul de l’horodatage                             */
/******************************************************************************/
tTIME_CalendarDate TIME_ComputeDate
(
    const tTIME_Calendar calendar,
    const tTIME_Timestamp timestamp
)
{
    tTIME_CalendarDate referenceDate;
    tTIME_CalendarDate currentDate;
    tTIME_LeapSeconds  leaps;

    switch(calendar)
    {
        case E_CALENDAR_JULIAN:
            referenceDate.calendar = E_CALENDAR_JULIAN;
            referenceDate.validity = true;
            referenceDate.year     = C_EPOCH_YEAR - 1;
            referenceDate.month    = E_MONTH_DECEMBER;
            referenceDate.day      = E_DAY_19;
            referenceDate.hour     = 0;
            referenceDate.minute   = 0;
            referenceDate.second   = 0;
            break;

        case E_CALENDAR_GREGORIAN:
            referenceDate.calendar = E_CALENDAR_GREGORIAN;
            referenceDate.validity = true;
            referenceDate.year     = C_EPOCH_YEAR;
            referenceDate.month    = E_MONTH_JANUARY;
            referenceDate.day      = E_DAY_01;
            referenceDate.hour     = 0;
            referenceDate.minute   = 0;
            referenceDate.second   = 0;
            break;

        case E_CALENDAR_UTC:
            referenceDate.calendar = E_CALENDAR_UTC;
            referenceDate.validity = true;
            referenceDate.year     = C_EPOCH_YEAR;
            referenceDate.month    = E_MONTH_JANUARY;
            referenceDate.day      = E_DAY_01;
            referenceDate.hour     = 0;
            referenceDate.minute   = 0;
            referenceDate.second   = 0;
            break;

        default:
            referenceDate.calendar = E_CALENDAR_UNKNOWN;
            referenceDate.validity = false;
            referenceDate.year     = C_UNDEFINED_DATE_YEAR;
            referenceDate.month    = C_UNDEFINED_DATE_MONTH;
            referenceDate.day      = C_UNDEFINED_DATE_DAY;
            referenceDate.hour     = C_UNDEFINED_DATE_HOUR;
            referenceDate.minute   = C_UNDEFINED_DATE_MINUTE;
            referenceDate.second   = C_UNDEFINED_DATE_SECOND;
            break;
    }

    currentDate = __TIME_ComputeDateBasedOnEpoch(&referenceDate, timestamp);

    if(calendar == E_CALENDAR_UTC)
    {
        for(leaps = 0 ; leaps < E_NB_LEAPSECONDS ; ++leaps)
        {
            if(timestamp == C_LEAPSECONDS[leaps])
            {
                currentDate.hour   = C_LEAPTIME_HOR;
                currentDate.minute = C_LEAPTIME_MIN;
                currentDate.second = C_LEAPTIME_SEC;

                if(currentDate.month == E_MONTH_JULY)
                {
                    currentDate.day   = E_DAY_30;
                    currentDate.month = E_MONTH_JUNE;
                }
                else
                {
                    currentDate.day   = E_DAY_31;
                    currentDate.month = E_MONTH_DECEMBER;
                    currentDate.year -= 1;
                }
            }
        }
    }

    return currentDate;
}

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
